'use strict';

/* --------------------------------------------------------------
 bulk_email_order.js 2016-05-25
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Bulk Email Order Modal Controller
 */
gx.controllers.module('bulk_email_order', [jse.source + '/vendor/momentjs/moment.min.js', 'modal', 'loading_spinner'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {
        bindings: { subject: $this.find('.subject') }
    };

    /**
     * Selector for the email list item.
     *
     * @type {String}
     */
    var emailListItemSelector = '.email-list-item';

    /**
     * Selector for the email list item ID.
     *
     * @type {String}
     */
    var emailListItemEmailSelector = '.email-input';

    /**
     * Selector for the modal content body layer.
     *
     * @type {String}
     */
    var modalContentSelector = '.modal-content';

    /**
     * Placeholder map.
     * Used to replace the placeholder with the respective variables.
     *
     * Format: '{Placeholder}' : 'Attribute'
     *
     * @type {Object}
     */
    var placeholderMap = {
        '{ORDER_ID}': 'id',
        '{ORDER_DATE}': 'purchaseDate'
    };

    /**
     * Loading spinner instance.
     *
     * @type {jQuery|null}
     */
    var $spinner = null;

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Show/hide loading spinner.
     *
     * @param {Boolean} doShow Show the loading spinner?
     */
    function _toggleSpinner(doShow) {
        if (doShow) {
            $spinner = jse.libs.loading_spinner.show($this.find(modalContentSelector), $this.css('z-index'));
        } else {
            jse.libs.loading_spinner.hide($spinner);
        }
    }

    /**
     * Parse subject and replace the placeholders with the variables.
     *
     * @param {Object} orderData Order data.
     *
     * @return {String}
     */
    function _getParsedSubject(orderData) {
        // Subject.
        var subject = module.bindings.subject.get();

        // Iterate over the placeholders and replace the values.
        Object.keys(placeholderMap).forEach(function (placeholder) {
            return subject = subject.replace(placeholder, orderData[placeholderMap[placeholder]]);
        });

        return subject;
    }

    /**
     * Handles the successful delivery of all messages.
     */
    function _handleDeliverySuccess() {
        var message = jse.core.lang.translate('BULK_MAIL_SUCCESS', 'gm_send_order');

        // Show success message in the admin info box.
        jse.libs.info_box.addSuccessMessage(message);

        // Hide modal and loading spinner.
        _toggleSpinner(false);
        $this.modal('hide');
    }

    /**
     * Handles the failure of the message delivery.
     */
    function _handleDeliveryFail() {
        var title = jse.core.lang.translate('error', 'messages');
        var content = jse.core.lang.translate('BULK_MAIL_UNSUCCESS', 'gm_send_order');

        // Show error message in a modal.
        jse.libs.modal.message({ title: title, content: content });

        // Hide modal and the loading spinner and reenable the send button.
        _toggleSpinner(false);
        $this.modal('hide');
    }

    /**
     * Send the modal data to the form through an AJAX call.
     */
    function _onSendClick() {
        // Send type.
        var REQUEST_SEND_TYPE = 'send_order';

        // Request base URL.
        var REQUEST_URL = jse.core.config.get('appUrl') + '/admin/gm_send_order.php';

        // Collection of requests in promise format.
        var promises = [];

        // Email list item elements.
        var $emailListItems = $this.find(emailListItemSelector);

        // Abort and hide modal on empty email list entries.
        if (!$emailListItems.length) {
            $this.modal('hide');
            return;
        }

        // Show loading spinner.
        _toggleSpinner(true);

        // Fill orders array with data.
        $emailListItems.each(function (index, element) {
            // Order data.
            var orderData = $(element).data('order');

            // Format the purchase date.
            var dateFormat = jse.core.config.get('languageCode') === 'de' ? 'DD.MM.YY' : 'MM.DD.YY';
            orderData.purchaseDate = moment(orderData.purchaseDate.date).format(dateFormat);

            // Email address entered in input field.
            var enteredEmail = $(element).find(emailListItemEmailSelector).val();

            // Request GET parameters to send.
            var getParameters = {
                oID: orderData.id,
                type: REQUEST_SEND_TYPE
            };

            // Composed request URL.
            var url = REQUEST_URL + '?' + $.param(getParameters);

            // Data to send.
            var data = {
                gm_mail: enteredEmail,
                gm_subject: _getParsedSubject(orderData)
            };

            // Promise wrapper for AJAX response.
            var promise = new Promise(function (resolve, reject) {
                // Create AJAX request.
                var request = $.ajax({ method: 'POST', url: url, data: data });

                request.done(function () {
                    var orderId = getParameters.oID;
                    var $tableRow = $('tbody tr#' + orderId);

                    // Remove the e-mail symbol
                    $tableRow.find('td.actions i.tooltip-confirmation-not-sent').remove();
                });

                // Resolve promise on success.
                request.done(function (response) {
                    return resolve(response);
                });

                // Reject promise on fail.
                request.fail(function () {
                    return reject();
                });
            });

            // Add promise to array.
            promises.push(promise);
        });

        // Wait for all promise to respond and handle success/error.
        Promise.all(promises).then(_handleDeliverySuccess).catch(_handleDeliveryFail);
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('click', '.btn.send', _onSendClick);
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
